using System;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using TMPro;
using UnityEngine;
using UnityEngine.XR.ARFoundation;

public class NavigationModule : MonoBehaviour
{
    [SerializeField] private LocationTracker locationTracker;
    [SerializeField] private PathVisualizer pathVisualizer;
    [SerializeField] private HTTPClient httpClient;
    [SerializeField] private CoordinateMapper coordinateMapper;
    [SerializeField] private ARTrackedImageManager trackedImageManager;
    
    /// <summary>
    /// The audio clip to play when a new tracked image is added.
    /// </summary>
    [SerializeField] private AudioClip trackedImageAlert;
    
    [SerializeField] private string apiBaseUrl = "https://api.arassistant.nl/navigation";
    [SerializeField] private string goalNodeId;

    
    /// <summary>
    /// The AudioSource used to play the alert sound.
    /// </summary>
    private AudioSource _audioSource;

    
    // private void OnEnable() => trackedImageManager.trackedImagesChanged += OnTrackedImagesChanged;
    // private void OnDisable() => trackedImageManager.trackedImagesChanged -= OnTrackedImagesChanged;
    private void OnEnable() => trackedImageManager.trackedImagesChanged += args => StartCoroutine(OnTrackedImagesChanged(args));
    private void OnDisable() => trackedImageManager.trackedImagesChanged -= args => StartCoroutine(OnTrackedImagesChanged(args));
    

    private void Awake()
    {
        _audioSource = gameObject.AddComponent<AudioSource>();
        _audioSource.clip = trackedImageAlert;
    }

    private Queue<string> nodeQueue = new();
    private string closestNodeId;
    private Vector3 currentTargetNodePosition;

    private Coroutine navigationCoroutine;

    [Serializable]
    class GraphInfo
    {
        public float originX;
        public float originY;
        public float originZ;
        public float scale;

        public Vector3 Origin
        {
            get { return new Vector3(originX, originY, originZ); }
        }
    }
    
    private IEnumerator OnTrackedImagesChanged(ARTrackedImagesChangedEventArgs args)
    {
        foreach (var trackedImage in args.added)
        {
            if (trackedImage.referenceImage.name != "0") continue;
            
            _audioSource?.Play();
            
            coordinateMapper.WorldOrigin = trackedImage.transform.position;

            yield return httpClient.GetRequest($"{apiBaseUrl}/graph/2", json =>
            {
                var graphInfo = JsonUtility.FromJson<GraphInfo>(json);
                Debug.Log("Got the graph info");
                coordinateMapper.GraphOrigin = new Vector3(graphInfo.originX, graphInfo.originY, graphInfo.originZ);
                coordinateMapper.GraphScale = graphInfo.scale;
            
                Debug.Log(coordinateMapper.ToString());
            });
            StartNavigation();
        }

        foreach (var trackedImage in args.updated)
        {
            if (trackedImage.referenceImage.name != "0") continue;
            coordinateMapper.WorldOrigin = trackedImage.transform.position;
        }

        foreach (var trackedImage in args.removed)
        {
            if (trackedImage.referenceImage.name != "0") continue;
        }
    }
    
    public void StartNavigation()
    {
        navigationCoroutine = StartCoroutine(NavigationRoutine());
    }

    public void StopNavigation()
    {
        if (navigationCoroutine != null)
        {
            StopCoroutine(navigationCoroutine);
            navigationCoroutine = null;
        }

        nodeQueue.Clear();
        pathVisualizer.ClearPath();
    }

    private IEnumerator NavigationRoutine()
    {
        // ----- Get closest node -----
        Vector3 currentLocation = coordinateMapper.ToModel(locationTracker.CurrentLocation);
        
        yield return httpClient.GetRequest($"{apiBaseUrl}/nearest?x={currentLocation.x}&y={currentLocation.y}&z={currentLocation.z}&to={goalNodeId}", (json) =>
        {
            closestNodeId = json;
            Debug.Log($"Closest node ID: {closestNodeId}");
        });

        // ---- Start Path -----
        yield return httpClient.GetRequest($"{apiBaseUrl}/path?from={closestNodeId}&to={goalNodeId}&type=shallow", (json) =>
        {
            Debug.Log(json);
            nodeQueue = new Queue<string>(JsonUtility.FromJson<NodeList>(json).nodes);
        });

        
        
        // loop through path
        while (nodeQueue.Count > 0)
        {
            string currentNodeId = nodeQueue.Peek();
            string nodeUrl = $"{apiBaseUrl}/node/{currentNodeId}";
            bool nodeLoaded = false;

            yield return httpClient.GetRequest(nodeUrl, (json) =>
            {
                var node = JsonUtility.FromJson<NodeData>(json);
                currentTargetNodePosition = coordinateMapper.ToWorld(new Vector3(node.x, node.y, node.z));
                pathVisualizer.ClearPath();
                pathVisualizer.PlaceNode(currentTargetNodePosition);
                pathVisualizer.DrawPath(locationTracker.CurrentLocation, currentTargetNodePosition);
                nodeLoaded = true;
            });

            if (!nodeLoaded) yield break;

            while (!locationTracker.IsNodeReached(currentTargetNodePosition))
            {
                yield return null;
            }

            nodeQueue.Dequeue();
        }

        // ----- End Path -----
        Debug.Log("Navigation complete. Destination reached.");
    }

    [System.Serializable]
    private class NodeList
    {
        public List<string> nodes;
    }
    
    [System.Serializable]
    private class NodeData
    {
        public float x, y, z;
    }
}
